<?php

namespace App\Http\Controllers\Karyawan;

use Log;
use Storage;
use Exception;
use Carbon\Carbon;
use App\Models\Absensi;
use App\Models\Karyawan;
use App\Models\Notifikasi;
use App\Models\LokasiKantor;
use Illuminate\Http\Request;
use App\Models\WajahKaryawan;
use App\Models\JadwalShift;
use App\Models\AjukanShift;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;

class AbsensiController extends Controller
{
    /**
     * Get shift aktif untuk karyawan berdasarkan tanggal
     */
    private function getActiveShift($karyawan, $tanggal)
    {
        $departemen_id = $karyawan->departemen_id;

        // 1. Cek apakah ada pengajuan shift SEMENTARA yang disetujui dan masih aktif
        $shiftSementara = AjukanShift::where('departemen_id', $departemen_id)
            ->where('jenis', 'sementara')
            ->where('status', 'disetujui')
            ->where('tanggal_mulai', '<=', $tanggal)
            ->where('tanggal_selesai', '>=', $tanggal)
            ->first();

        if ($shiftSementara) {
            return $shiftSementara->shiftBaru; // Return shift pengganti
        }

        // 2. Jika tidak ada shift sementara, ambil shift default dari jadwal_shift yang aktif
        $jadwalShift = JadwalShift::where('departemen_id', $departemen_id)
            ->where('is_active', true)
            ->where('tanggal_mulai', '<=', $tanggal)
            ->where(function ($q) use ($tanggal) {
                $q->whereNull('tanggal_selesai')
                ->orWhere('tanggal_selesai', '>=', $tanggal);
            })
            ->orderBy('tanggal_mulai', 'desc')
            ->first();


        if ($jadwalShift) {
            return $jadwalShift->shift;
        }

        return null;
    }

    /**
     * Show form absen masuk
     */
    public function masukForm()
    {
        $karyawan = Karyawan::where('user_id', Auth::id())->firstOrFail();

        // Cek apakah sudah absen masuk hari ini
        $today = date('Y-m-d');
        $existingAbsen = Absensi::where('karyawan_id', $karyawan->id)
            ->whereDate('tanggal', $today)
            ->first();

        if ($existingAbsen && $existingAbsen->jam_masuk) {
            return redirect()->route('karyawan.dashboard')
                ->with('error', 'Anda sudah absen masuk hari ini!');
        }

        // Ambil data wajah karyawan
        $wajahKaryawan = WajahKaryawan::where('karyawan_id', $karyawan->id)->first();

        if (!$wajahKaryawan) {
            return redirect()->route('karyawan.dashboard')
                ->with('error', 'Wajah Anda belum terdaftar! Hubungi admin untuk registrasi.');
        }

        // Get shift aktif
        $shift = $this->getActiveShift($karyawan, $today);

        if (!$shift) {
            return redirect()->route('karyawan.dashboard')
                ->with('error', 'Shift departemen Anda belum ditentukan. Hubungi admin!');
        }

        return view('karyawan.absensi.masuk-form', compact('wajahKaryawan', 'shift'));
    }

    /**
     * Process absen masuk
     */
    public function masuk(Request $request)
    {
        try {
            $request->validate([
                'latitude' => 'required|numeric',
                'longitude' => 'required|numeric',
                'face_encoding' => 'required|json',
                'face_image' => 'required',
                'face_confidence' => 'required|numeric'
            ]);

            $karyawan = Karyawan::where('user_id', Auth::id())->firstOrFail();
            $today = date('Y-m-d');

            // Cek sudah absen
            $existingAbsen = Absensi::where('karyawan_id', $karyawan->id)
                ->whereDate('tanggal', $today)
                ->first();

            if ($existingAbsen && $existingAbsen->jam_masuk) {
                return redirect()->route('karyawan.dashboard')
                    ->with('error', 'Anda sudah absen masuk hari ini!');
            }

            // Validasi lokasi
            $lokasi = LokasiKantor::first();
            $distance = $this->calculateDistance(
                $request->latitude,
                $request->longitude,
                $lokasi->latitude,
                $lokasi->longitude
            );

            if ($distance > $lokasi->radius) {
                return back()->with('error', 'Anda berada di luar radius kantor! Jarak: ' . round($distance) . 'm');
            }

            // Validasi face
            $wajahTerdaftar = WajahKaryawan::where('karyawan_id', $karyawan->id)->first();
            if (!$wajahTerdaftar) {
                return back()->with('error', 'Wajah Anda belum terdaftar!');
            }

            $encodingBaru = json_decode($request->face_encoding);
            $encodingTerdaftar = json_decode($wajahTerdaftar->face_encoding);
            $faceDistance = $this->euclideanDistance($encodingBaru, $encodingTerdaftar);

            if ($faceDistance >= 0.6) {
                return back()->with('error', 'Wajah tidak cocok! Distance: ' . round($faceDistance, 3));
            }

            // Simpan foto ke storage
            $fotoPath = null;
            if ($request->face_image) {
                $image = $request->face_image;
                $image = str_replace('data:image/jpeg;base64,', '', $image);
                $image = str_replace(' ', '+', $image);
                $imageData = base64_decode($image);

                $fileName = 'absen_masuk_' . $karyawan->id . '_' . date('Ymd_His') . '.jpg';
                Storage::disk('public')->put('absensi/' . $fileName, $imageData);
                $fotoPath = 'absensi/' . $fileName;
            }

            // Get shift aktif untuk hari ini
            $shift = $this->getActiveShift($karyawan, $today);

            if (!$shift) {
                return back()->with('error', 'Shift departemen anda belum ditentukan. Hubungi admin!');
            }

            $now = Carbon::now();
            $jamMasuk = $now->format('H:i:s');

            // Jam masuk shift hari ini
            $jamMasukShift = Carbon::parse($today . ' ' . $shift->jam_masuk);

            // Toleransi keterlambatan
            $batasTerlambat = $jamMasukShift->copy()->addMinutes($shift->toleransi_menit);

            $status = $now->lte($batasTerlambat) ? 'hadir' : 'terlambat';

            // Simpan absensi
            Absensi::create([
                'karyawan_id' => $karyawan->id,
                'tanggal' => $today,
                'jam_masuk' => $jamMasuk,
                'latitude' => $request->latitude,
                'longitude' => $request->longitude,
                'foto_masuk' => $fotoPath,
                'face_valid' => 1,
                'face_confidence' => $request->face_confidence,
                'face_distance' => $faceDistance,
                'verification_method' => 'face',
                'status' => $status
            ]);

            activity_log(
                'absensi',
                'clock_in',
                'Absen masuk pukul ' . now()->format('H:i')
            );

            Notifikasi::create([
                'user_id' => Auth::id(),
                'judul' => 'Absen Berhasil',
                'pesan' => 'Anda berhasil melakukan absen masuk hari ini.',
                'type' => 'absensi',
                'target_role' => 'karyawan'
            ]);

            $message = $status == 'hadir'
                ? '✔ Absen masuk berhasil! Anda tepat waktu.'
                : '⚠ Absen masuk berhasil, tetapi Anda terlambat.';

            return redirect()->route('karyawan.dashboard')->with('success', $message);

        } catch (Exception $e) {
            Log::error('Absen Masuk Error: ' . $e->getMessage());
            return back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage());
        }
    }

    /**
     * Show form absen pulang
     */
    public function pulangForm()
    {
        $karyawan = Karyawan::where('user_id', Auth::id())->firstOrFail();

        // Cek apakah sudah absen masuk hari ini
        $today = date('Y-m-d');
        $absensi = Absensi::where('karyawan_id', $karyawan->id)
            ->whereDate('tanggal', $today)
            ->first();

        // Jika tidak ada absensi hari ini, cek kemarin (untuk shift lintas hari)
        if (!$absensi) {
            $yesterday = Carbon::yesterday()->toDateString();
            $shift = $this->getActiveShift($karyawan, $yesterday);

            if ($shift && $shift->lintas_hari) {
                $absensi = Absensi::where('karyawan_id', $karyawan->id)
                    ->whereDate('tanggal', $yesterday)
                    ->first();
            }
        }

        if (!$absensi || !$absensi->jam_masuk) {
            return redirect()->route('karyawan.dashboard')
                ->with('error', 'Anda belum absen masuk hari ini!');
        }

        if ($absensi->jam_pulang) {
            return redirect()->route('karyawan.dashboard')
                ->with('error', 'Anda sudah absen pulang hari ini!');
        }

        // Ambil data wajah karyawan
        $wajahKaryawan = WajahKaryawan::where('karyawan_id', $karyawan->id)->first();

        if (!$wajahKaryawan) {
            return redirect()->route('karyawan.dashboard')
                ->with('error', 'Wajah Anda belum terdaftar! Hubungi admin untuk registrasi.');
        }

        return view('karyawan.absensi.pulang-form', compact('wajahKaryawan'));
    }

    /**
     * Process absen pulang
     */
    public function pulang(Request $request)
    {
        try {
            $request->validate([
                'latitude' => 'required|numeric',
                'longitude' => 'required|numeric',
                'face_encoding' => 'required|json',
                'face_image' => 'required',
                'face_confidence' => 'required|numeric'
            ]);

            $karyawan = Karyawan::where('user_id', Auth::id())->firstOrFail();
            $today = Carbon::now()->toDateString();

            // Cari absen hari ini
            $absensi = Absensi::where('karyawan_id', $karyawan->id)
                ->whereDate('tanggal', $today)
                ->first();

            // Jika tidak ada, cek kemarin untuk shift lintas hari
            if (!$absensi) {
                $yesterday = Carbon::yesterday()->toDateString();
                $shift = $this->getActiveShift($karyawan, $yesterday);

                if ($shift && $shift->lintas_hari) {
                    $absensi = Absensi::where('karyawan_id', $karyawan->id)
                        ->whereDate('tanggal', $yesterday)
                        ->first();
                }
            }

            if (!$absensi || !$absensi->jam_masuk) {
                return redirect()->route('karyawan.dashboard')
                    ->with('error', 'Anda belum absen masuk hari ini!');
            }

            if ($absensi->jam_pulang) {
                return redirect()->route('karyawan.dashboard')
                    ->with('error', 'Anda sudah absen pulang hari ini!');
            }

            // Validasi lokasi
            $lokasi = LokasiKantor::first();
            $distance = $this->calculateDistance(
                $request->latitude,
                $request->longitude,
                $lokasi->latitude,
                $lokasi->longitude
            );

            if ($distance > $lokasi->radius) {
                return back()->with(
                    'error',
                    'Anda berada di luar radius kantor! Jarak: ' . round($distance) . ' meter'
                );
            }

            // Validasi face
            $wajahTerdaftar = WajahKaryawan::where('karyawan_id', $karyawan->id)->first();
            if (!$wajahTerdaftar) {
                return back()->with('error', 'Wajah Anda belum terdaftar!');
            }

            $encodingBaru = json_decode($request->face_encoding);
            $encodingTerdaftar = json_decode($wajahTerdaftar->face_encoding);
            $faceDistance = $this->euclideanDistance($encodingBaru, $encodingTerdaftar);

            if ($faceDistance >= 0.6) {
                return back()->with(
                    'error',
                    'Wajah tidak cocok! Distance: ' . round($faceDistance, 3)
                );
            }

            // Simpan foto pulang ke storage
            $fotoPath = null;
            if ($request->face_image) {
                $image = str_replace('data:image/jpeg;base64,', '', $request->face_image);
                $image = str_replace(' ', '+', $image);
                $imageData = base64_decode($image);

                $fileName = 'absen_pulang_' . $karyawan->id . '_' . Carbon::now()->format('Ymd_His') . '.jpg';
                Storage::disk('public')->put('absensi/' . $fileName, $imageData);
                $fotoPath = 'absensi/' . $fileName;
            }

            // Update absensi
            $absensi->update([
                'jam_pulang' => Carbon::now()->format('H:i:s'),
                'foto_pulang' => $fotoPath,
                'face_confidence' => $request->face_confidence,
                'face_distance' => $faceDistance
            ]);

            activity_log(
                'absensi',
                'clock_out',
                'Absen pulang pukul ' . now()->format('H:i')
            );

            Notifikasi::create([
                'user_id' => Auth::id(),
                'judul' => 'Absen Pulang Berhasil',
                'pesan' => 'Anda berhasil melakukan absen pulang. Hati-hati di jalan!',
                'type' => 'absensi',
                'target_role' => 'karyawan'
            ]);

            return redirect()->route('karyawan.dashboard')
                ->with('success', '✔ Absen pulang berhasil! Hati-hati di jalan.');

        } catch (Exception $e) {
            Log::error('Absen Pulang Error: ' . $e->getMessage());
            return back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage());
        }
    }

    /**
     * Show attendance log
     */
    public function log(Request $request)
    {
        $karyawan = Karyawan::where('user_id', auth()->id())->firstOrFail();
        $bulan = $request->bulan ?? now()->format('Y-m');

        $absensi = Absensi::where('karyawan_id', $karyawan->id)
            ->whereMonth('tanggal', Carbon::parse($bulan)->month)
            ->whereYear('tanggal', Carbon::parse($bulan)->year)
            ->orderBy('tanggal', 'desc')
            ->get();

        return view('karyawan.absensi.log', compact('absensi', 'bulan'));
    }

    /**
     * Calculate distance between two GPS coordinates (Haversine formula)
     */
    private function calculateDistance($lat1, $lon1, $lat2, $lon2)
    {
        $earthRadius = 6371000; // meters

        $dLat = deg2rad($lat2 - $lat1);
        $dLon = deg2rad($lon2 - $lon1);

        $a = sin($dLat / 2) * sin($dLat / 2) +
            cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
            sin($dLon / 2) * sin($dLon / 2);

        $c = 2 * atan2(sqrt($a), sqrt(1 - $a));

        return $earthRadius * $c;
    }

    /**
     * Euclidean distance for face matching
     */
    private function euclideanDistance($arr1, $arr2)
    {
        $sum = 0;
        for ($i = 0; $i < count($arr1); $i++) {
            $sum += pow($arr1[$i] - $arr2[$i], 2);
        }
        return sqrt($sum);
    }

    /**
 * Show face registration form for karyawan
 */
public function registerFaceForm()
{
    $karyawan = Karyawan::where('user_id', Auth::id())->firstOrFail();

    // Cek apakah wajah sudah terdaftar
    // $wajahKaryawan = WajahKaryawan::where('karyawan_id', $karyawan->id)->first();

    // if ($wajahKaryawan) {
    //     return redirect()->route('karyawan.dashboard')
    //         ->with('success', 'Berhasil mendaftarkan wajah');
    // }

    return view('karyawan.wajah.register');
}

/**
 * Store face registration for karyawan
 */
public function registerFaceStore(Request $request)
{
    try {
        $request->validate([
            'face_encoding' => 'required|json',
            'face_image' => 'required',
            'face_confidence' => 'required|numeric'
        ]);

        $karyawan = Karyawan::where('user_id', Auth::id())->firstOrFail();

        // Cek apakah sudah pernah daftar
        $existingFace = WajahKaryawan::where('karyawan_id', $karyawan->id)->first();

        if ($existingFace) {
            return redirect()->route('karyawan.dashboard')
                ->with('error', 'Wajah Anda sudah terdaftar! Hubungi admin jika ingin update.');
        }

        // Simpan foto wajah ke storage
        $fotoPath = null;
        if ($request->face_image) {
            $image = $request->face_image;
            $image = str_replace('data:image/jpeg;base64,', '', $image);
            $image = str_replace(' ', '+', $image);
            $imageData = base64_decode($image);

            $fileName = 'face_' . $karyawan->id . '_' . time() . '.jpg';
            Storage::disk('public')->put('faces/' . $fileName, $imageData);
            $fotoPath = 'faces/' . $fileName;
        }

        // Simpan data wajah
        WajahKaryawan::create([
            'karyawan_id' => $karyawan->id,
            'face_encoding' => $request->face_encoding,
            'face_image' => $fotoPath,
            'registered_at' => now(),
            'registered_by' => Auth::id()
        ]);

        // Update status wajah terdaftar
        $karyawan->update(['wajah_terdaftar' => 1]);

        // Activity log
        activity_log(
            'wajah',
            'register',
            'Berhasil mendaftarkan wajah untuk absensi'
        );

        // Notifikasi
        Notifikasi::create([
            'user_id' => Auth::id(),
            'judul' => 'Wajah Berhasil Didaftarkan',
            'pesan' => 'Wajah Anda berhasil terdaftar! Sekarang Anda bisa melakukan absensi dengan face recognition.',
            'type' => 'wajah',
            'target_role' => 'karyawan'
        ]);

        return redirect()->route('karyawan.dashboard')
            ->with('success', '✅ Wajah berhasil didaftarkan! Sekarang Anda bisa absen dengan face recognition.');

    } catch (Exception $e) {
        Log::error('Face Registration Error: ' . $e->getMessage());
        return back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage());
    }
}

}
